"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ForgeGradle3Adapter = void 0;
const forge_resolver_1 = require("../forge.resolver");
const LoggerUtil_1 = require("../../../util/LoggerUtil");
const versionutil_1 = require("../../../util/versionutil");
const helios_distribution_types_1 = require("helios-distribution-types");
const LibRepo_struct_1 = require("../../../structure/repo/LibRepo.struct");
const fs_extra_1 = require("fs-extra");
const path_1 = require("path");
const child_process_1 = require("child_process");
const javautil_1 = require("../../../util/java/javautil");
const maven_1 = require("../../../util/maven");
const crypto_1 = require("crypto");
class ForgeGradle3Adapter extends forge_resolver_1.ForgeResolver {
    constructor(absoluteRoot, relativeRoot, baseUrl, minecraftVersion, forgeVersion, discardOutput, invalidateCache) {
        super(absoluteRoot, relativeRoot, baseUrl, minecraftVersion, forgeVersion, discardOutput, invalidateCache);
        this.configure();
    }
    static isForVersion(version, libraryVersion) {
        if (version.getMinor() === 12 && versionutil_1.VersionUtil.isOneDotTwelveFG2(libraryVersion)) {
            return false;
        }
        return versionutil_1.VersionUtil.isVersionAcceptable(version, [12, 13, 14, 15, 16]);
    }
    configure() {
        // Configure for 13, 14, 15, 16
        if (versionutil_1.VersionUtil.isVersionAcceptable(this.minecraftVersion, [13, 14, 15, 16])) {
            // https://github.com/MinecraftForge/MinecraftForge/commit/97d4652f5fe15931b980117efabdff332f9f6428
            const mcpUnifiedVersion = `${this.minecraftVersion}-${ForgeGradle3Adapter.WILDCARD_MCP_VERSION}`;
            this.generatedFiles = [
                {
                    name: 'base jar',
                    group: LibRepo_struct_1.LibRepoStructure.FORGE_GROUP,
                    artifact: LibRepo_struct_1.LibRepoStructure.FORGE_ARTIFACT,
                    version: this.artifactVersion,
                    classifiers: [undefined]
                },
                {
                    name: 'universal jar',
                    group: LibRepo_struct_1.LibRepoStructure.FORGE_GROUP,
                    artifact: LibRepo_struct_1.LibRepoStructure.FORGE_ARTIFACT,
                    version: this.artifactVersion,
                    classifiers: ['universal']
                },
                {
                    name: 'client jar',
                    group: LibRepo_struct_1.LibRepoStructure.FORGE_GROUP,
                    artifact: LibRepo_struct_1.LibRepoStructure.FORGE_ARTIFACT,
                    version: this.artifactVersion,
                    classifiers: ['client']
                },
                {
                    name: 'client data',
                    group: LibRepo_struct_1.LibRepoStructure.MINECRAFT_GROUP,
                    artifact: LibRepo_struct_1.LibRepoStructure.MINECRAFT_CLIENT_ARTIFACT,
                    version: this.minecraftVersion.toString(),
                    classifiers: ['data'],
                    skipIfNotPresent: true
                },
                {
                    name: 'client srg',
                    group: LibRepo_struct_1.LibRepoStructure.MINECRAFT_GROUP,
                    artifact: LibRepo_struct_1.LibRepoStructure.MINECRAFT_CLIENT_ARTIFACT,
                    version: mcpUnifiedVersion,
                    classifiers: ['srg']
                }
            ];
            this.wildcardsInUse = [
                ForgeGradle3Adapter.WILDCARD_MCP_VERSION
            ];
            if (versionutil_1.VersionUtil.isVersionAcceptable(this.minecraftVersion, [13, 14, 15])) {
                this.generatedFiles.push({
                    name: 'client slim',
                    group: LibRepo_struct_1.LibRepoStructure.MINECRAFT_GROUP,
                    artifact: LibRepo_struct_1.LibRepoStructure.MINECRAFT_CLIENT_ARTIFACT,
                    version: this.minecraftVersion.toString(),
                    classifiers: [
                        'slim',
                        'slim-stable'
                    ]
                }, {
                    name: 'client extra',
                    group: LibRepo_struct_1.LibRepoStructure.MINECRAFT_GROUP,
                    artifact: LibRepo_struct_1.LibRepoStructure.MINECRAFT_CLIENT_ARTIFACT,
                    version: this.minecraftVersion.toString(),
                    classifiers: [
                        'extra',
                        'extra-stable'
                    ]
                });
            }
            else {
                this.generatedFiles.push({
                    name: 'client slim',
                    group: LibRepo_struct_1.LibRepoStructure.MINECRAFT_GROUP,
                    artifact: LibRepo_struct_1.LibRepoStructure.MINECRAFT_CLIENT_ARTIFACT,
                    version: mcpUnifiedVersion,
                    classifiers: [
                        'slim',
                        'slim-stable'
                    ]
                }, {
                    name: 'client extra',
                    group: LibRepo_struct_1.LibRepoStructure.MINECRAFT_GROUP,
                    artifact: LibRepo_struct_1.LibRepoStructure.MINECRAFT_CLIENT_ARTIFACT,
                    version: mcpUnifiedVersion,
                    classifiers: [
                        'extra',
                        'extra-stable'
                    ]
                });
            }
            return;
        }
        // Configure for 12
        if (versionutil_1.VersionUtil.isVersionAcceptable(this.minecraftVersion, [12])) {
            // NOTHING TO CONFIGURE
            return;
        }
    }
    async getModule() {
        return this.process();
    }
    isForVersion(version, libraryVersion) {
        return ForgeGradle3Adapter.isForVersion(version, libraryVersion);
    }
    async process() {
        const libRepo = this.repoStructure.getLibRepoStruct();
        // Get Installer
        const installerPath = libRepo.getLocalForge(this.artifactVersion, 'installer');
        ForgeGradle3Adapter.logger.debug(`Checking for forge installer at ${installerPath}..`);
        if (!await libRepo.artifactExists(installerPath)) {
            ForgeGradle3Adapter.logger.debug('Forge installer not found locally, initializing download..');
            await libRepo.downloadArtifactByComponents(this.REMOTE_REPOSITORY, LibRepo_struct_1.LibRepoStructure.FORGE_GROUP, LibRepo_struct_1.LibRepoStructure.FORGE_ARTIFACT, this.artifactVersion, 'installer', 'jar');
        }
        else {
            ForgeGradle3Adapter.logger.debug('Using locally discovered forge installer.');
        }
        ForgeGradle3Adapter.logger.debug(`Beginning processing of Forge v${this.forgeVersion} (Minecraft ${this.minecraftVersion})`);
        if (this.generatedFiles != null && this.generatedFiles.length > 0) {
            // Run installer
            return this.processWithInstaller(installerPath);
        }
        else {
            // Installer not required
            return this.processWithoutInstaller(installerPath);
        }
    }
    async processWithInstaller(installerPath) {
        let doInstall = true;
        // Check cache.
        const cacheDir = this.repoStructure.getForgeCacheDirectory(this.artifactVersion);
        if (await (0, fs_extra_1.pathExists)(cacheDir)) {
            if (this.invalidateCache) {
                ForgeGradle3Adapter.logger.info(`Removing existing cache ${cacheDir}..`);
                await (0, fs_extra_1.remove)(cacheDir);
            }
            else {
                // Use cache.
                doInstall = false;
                ForgeGradle3Adapter.logger.info(`Using cached results at ${cacheDir}.`);
            }
        }
        else {
            await (0, fs_extra_1.mkdirs)(cacheDir);
        }
        const installerOutputDir = cacheDir;
        if (doInstall) {
            const workingInstaller = (0, path_1.join)(installerOutputDir, (0, path_1.basename)(installerPath));
            await (0, fs_extra_1.copy)(installerPath, workingInstaller);
            // Required for the installer to function.
            await (0, fs_extra_1.writeFile)((0, path_1.join)(installerOutputDir, 'launcher_profiles.json'), JSON.stringify({}));
            ForgeGradle3Adapter.logger.debug('Spawning forge installer');
            ForgeGradle3Adapter.logger.info('============== [ IMPORTANT ] ==============');
            ForgeGradle3Adapter.logger.info('When the installer opens please set the client installation directory to:');
            ForgeGradle3Adapter.logger.info(installerOutputDir);
            ForgeGradle3Adapter.logger.info('===========================================');
            await this.executeInstaller(workingInstaller);
            ForgeGradle3Adapter.logger.debug('Installer finished, beginning processing..');
        }
        ForgeGradle3Adapter.logger.debug('Processing Version Manifest');
        const versionManifestTuple = await this.processVersionManifest(installerOutputDir);
        const versionManifest = versionManifestTuple[0];
        ForgeGradle3Adapter.logger.debug('Processing generated forge files.');
        const forgeModule = await this.processForgeModule(versionManifest, installerOutputDir);
        // Attach version.json module.
        forgeModule.subModules?.unshift(versionManifestTuple[1]);
        ForgeGradle3Adapter.logger.debug('Processing Libraries');
        const libs = await this.processLibraries(versionManifest, installerOutputDir);
        forgeModule.subModules = forgeModule.subModules?.concat(libs);
        if (this.discardOutput) {
            ForgeGradle3Adapter.logger.info(`Removing installer output at ${installerOutputDir}..`);
            await (0, fs_extra_1.remove)(installerOutputDir);
            ForgeGradle3Adapter.logger.info('Removed successfully.');
        }
        return forgeModule;
    }
    async processVersionManifest(installerOutputDir) {
        const versionRepo = this.repoStructure.getVersionRepoStruct();
        const versionName = versionRepo.getFileName(this.minecraftVersion, this.forgeVersion);
        const versionManifestPath = (0, path_1.join)(installerOutputDir, 'versions', versionName, `${versionName}.json`);
        const versionManifestBuf = await (0, fs_extra_1.readFile)(versionManifestPath);
        const versionManifest = JSON.parse(versionManifestBuf.toString());
        const versionManifestModule = {
            id: this.artifactVersion,
            name: 'Minecraft Forge (version.json)',
            type: helios_distribution_types_1.Type.VersionManifest,
            artifact: this.generateArtifact(versionManifestBuf, await (0, fs_extra_1.lstat)(versionManifestPath), versionRepo.getVersionManifestURL(this.baseUrl, this.minecraftVersion, this.forgeVersion))
        };
        const destination = versionRepo.getVersionManifest(this.minecraftVersion, this.forgeVersion);
        await (0, fs_extra_1.copy)(versionManifestPath, destination, { overwrite: true });
        return [versionManifest, versionManifestModule];
    }
    async processForgeModule(versionManifest, installerOutputDir) {
        const libDir = (0, path_1.join)(installerOutputDir, 'libraries');
        if (this.wildcardsInUse) {
            if (this.wildcardsInUse.indexOf(ForgeGradle3Adapter.WILDCARD_MCP_VERSION) > -1) {
                const mcpVersion = this.getMCPVersion(versionManifest.arguments.game);
                if (mcpVersion == null) {
                    throw new Error('MCP Version not found.. did forge change their format?');
                }
                this.generatedFiles = this.generatedFiles.map(f => {
                    if (f.version.indexOf(ForgeGradle3Adapter.WILDCARD_MCP_VERSION) > -1) {
                        return {
                            ...f,
                            version: f.version.replace(ForgeGradle3Adapter.WILDCARD_MCP_VERSION, mcpVersion)
                        };
                    }
                    return f;
                });
            }
        }
        const mdls = [];
        for (const entry of this.generatedFiles) {
            const targetLocations = [];
            let located = false;
            classifierLoop: for (const _classifier of entry.classifiers) {
                const targetLocalPath = (0, path_1.join)(libDir, maven_1.MavenUtil.mavenComponentsToPath(entry.group, entry.artifact, entry.version, _classifier));
                targetLocations.push(targetLocalPath);
                const exists = await (0, fs_extra_1.pathExists)(targetLocalPath);
                if (exists) {
                    mdls.push({
                        id: maven_1.MavenUtil.mavenComponentsToIdentifier(entry.group, entry.artifact, entry.version, _classifier),
                        name: `Minecraft Forge (${entry.name})`,
                        type: helios_distribution_types_1.Type.Library,
                        artifact: this.generateArtifact(await (0, fs_extra_1.readFile)(targetLocalPath), await (0, fs_extra_1.lstat)(targetLocalPath), this.repoStructure.getLibRepoStruct().getArtifactUrlByComponents(this.baseUrl, entry.group, entry.artifact, entry.version, _classifier)),
                        subModules: []
                    });
                    const destination = this.repoStructure.getLibRepoStruct().getArtifactByComponents(entry.group, entry.artifact, entry.version, _classifier);
                    await (0, fs_extra_1.copy)(targetLocalPath, destination, { overwrite: true });
                    located = true;
                    break classifierLoop;
                }
            }
            if (!entry.skipIfNotPresent && !located) {
                throw new Error(`Required file ${entry.name} not found at any expected location:\n\t${targetLocations.join('\n\t')}`);
            }
        }
        const forgeModule = mdls.shift();
        forgeModule.type = helios_distribution_types_1.Type.ForgeHosted;
        forgeModule.subModules = mdls;
        return forgeModule;
    }
    async processLibraries(manifest, installerOutputDir) {
        const libDir = (0, path_1.join)(installerOutputDir, 'libraries');
        const libRepo = this.repoStructure.getLibRepoStruct();
        const mdls = [];
        for (const entry of manifest.libraries) {
            const artifact = entry.downloads.artifact;
            if (artifact.url) {
                const targetLocalPath = (0, path_1.join)(libDir, artifact.path);
                if (!await (0, fs_extra_1.pathExists)(targetLocalPath)) {
                    throw new Error(`Expected library ${entry.name} not found!`);
                }
                const components = maven_1.MavenUtil.getMavenComponents(entry.name);
                mdls.push({
                    id: entry.name,
                    name: `Minecraft Forge (${components.artifact})`,
                    type: helios_distribution_types_1.Type.Library,
                    artifact: this.generateArtifact(await (0, fs_extra_1.readFile)(targetLocalPath), await (0, fs_extra_1.lstat)(targetLocalPath), libRepo.getArtifactUrlByComponents(this.baseUrl, components.group, components.artifact, components.version, components.classifier, components.extension))
                });
                const destination = libRepo.getArtifactByComponents(components.group, components.artifact, components.version, components.classifier, components.extension);
                await (0, fs_extra_1.copy)(targetLocalPath, destination, { overwrite: true });
            }
        }
        return mdls;
    }
    executeInstaller(installerExec) {
        return new Promise(resolve => {
            const fiLogger = LoggerUtil_1.LoggerUtil.getLogger('Forge Installer');
            const child = (0, child_process_1.spawn)(javautil_1.JavaUtil.getJavaExecutable(), [
                '-jar',
                installerExec
            ], {
                cwd: (0, path_1.dirname)(installerExec)
            });
            child.stdout.on('data', (data) => fiLogger.info(data.toString('utf8').trim()));
            child.stderr.on('data', (data) => fiLogger.error(data.toString('utf8').trim()));
            child.on('close', code => {
                if (code === 0) {
                    fiLogger.info('Exited with code', code);
                }
                else {
                    fiLogger.error('Exited with code', code);
                }
                resolve();
            });
        });
    }
    getMCPVersion(args) {
        for (let i = 0; i < args.length; i++) {
            if (args[i] === '--fml.mcpVersion') {
                return args[i + 1];
            }
        }
        return null;
    }
    async processWithoutInstaller(installerPath) {
        // Extract version.json from installer.
        let versionManifestBuf;
        try {
            versionManifestBuf = await this.getVersionManifestFromJar(installerPath);
        }
        catch (err) {
            throw new Error('Failed to find version.json in forge installer jar.');
        }
        const versionManifest = JSON.parse(versionManifestBuf.toString());
        // Save Version Manifest
        const versionManifestDest = this.repoStructure.getVersionRepoStruct().getVersionManifest(this.minecraftVersion, this.forgeVersion);
        await (0, fs_extra_1.mkdirs)((0, path_1.dirname)(versionManifestDest));
        await (0, fs_extra_1.writeJson)(versionManifestDest, versionManifest, { spaces: 4 });
        const libRepo = this.repoStructure.getLibRepoStruct();
        const universalLocalPath = libRepo.getLocalForge(this.artifactVersion, 'universal');
        ForgeGradle3Adapter.logger.debug(`Checking for Forge Universal jar at ${universalLocalPath}..`);
        const forgeMdl = versionManifest.libraries.find(val => val.name.startsWith('net.minecraftforge:forge:'));
        if (forgeMdl == null) {
            throw new Error('Forge entry not found in version.json!');
        }
        let forgeUniversalBuffer;
        // Check for local universal jar.
        if (await libRepo.artifactExists(universalLocalPath)) {
            const localUniBuf = await (0, fs_extra_1.readFile)(universalLocalPath);
            const sha1 = (0, crypto_1.createHash)('sha1').update(localUniBuf).digest('hex');
            if (sha1 !== forgeMdl.downloads.artifact.sha1) {
                ForgeGradle3Adapter.logger.debug('SHA-1 of local universal jar does not match version.json entry.');
                ForgeGradle3Adapter.logger.debug('Redownloading Forge Universal jar..');
            }
            else {
                ForgeGradle3Adapter.logger.debug('Using locally discovered forge.');
                forgeUniversalBuffer = localUniBuf;
            }
        }
        else {
            ForgeGradle3Adapter.logger.debug('Forge Universal jar not found locally, initializing download..');
        }
        // Download if local is missing or corrupt
        if (!forgeUniversalBuffer) {
            await libRepo.downloadArtifactByComponents(this.REMOTE_REPOSITORY, LibRepo_struct_1.LibRepoStructure.FORGE_GROUP, LibRepo_struct_1.LibRepoStructure.FORGE_ARTIFACT, this.artifactVersion, 'universal', 'jar');
            forgeUniversalBuffer = await (0, fs_extra_1.readFile)(universalLocalPath);
        }
        ForgeGradle3Adapter.logger.debug(`Beginning processing of Forge v${this.forgeVersion} (Minecraft ${this.minecraftVersion})`);
        const forgeModule = {
            id: maven_1.MavenUtil.mavenComponentsToIdentifier(LibRepo_struct_1.LibRepoStructure.FORGE_GROUP, LibRepo_struct_1.LibRepoStructure.FORGE_ARTIFACT, this.artifactVersion, 'universal'),
            name: 'Minecraft Forge',
            type: helios_distribution_types_1.Type.ForgeHosted,
            artifact: this.generateArtifact(forgeUniversalBuffer, await (0, fs_extra_1.lstat)(universalLocalPath), libRepo.getArtifactUrlByComponents(this.baseUrl, LibRepo_struct_1.LibRepoStructure.FORGE_GROUP, LibRepo_struct_1.LibRepoStructure.FORGE_ARTIFACT, this.artifactVersion, 'universal')),
            subModules: []
        };
        // Attach Version Manifest module.
        forgeModule.subModules?.push({
            id: this.artifactVersion,
            name: 'Minecraft Forge (version.json)',
            type: helios_distribution_types_1.Type.VersionManifest,
            artifact: this.generateArtifact(await (0, fs_extra_1.readFile)(versionManifestDest), await (0, fs_extra_1.lstat)(versionManifestDest), this.repoStructure.getVersionRepoStruct().getVersionManifestURL(this.baseUrl, this.minecraftVersion, this.forgeVersion))
        });
        for (const lib of versionManifest.libraries) {
            if (lib.name.startsWith('net.minecraftforge:forge:')) {
                // We've already processed forge.
                continue;
            }
            ForgeGradle3Adapter.logger.debug(`Processing ${lib.name}..`);
            const extension = 'jar';
            const localPath = libRepo.getArtifactById(lib.name, extension);
            let queueDownload = !await libRepo.artifactExists(localPath);
            let libBuf;
            if (!queueDownload) {
                libBuf = await (0, fs_extra_1.readFile)(localPath);
                const sha1 = (0, crypto_1.createHash)('sha1').update(libBuf).digest('hex');
                if (sha1 !== lib.downloads.artifact.sha1) {
                    ForgeGradle3Adapter.logger.debug('Hashes do not match, redownloading..');
                    queueDownload = true;
                }
            }
            else {
                ForgeGradle3Adapter.logger.debug('Not found locally, downloading..');
                queueDownload = true;
            }
            if (queueDownload) {
                await libRepo.downloadArtifactDirect(lib.downloads.artifact.url, lib.downloads.artifact.path);
                libBuf = await (0, fs_extra_1.readFile)(localPath);
            }
            else {
                ForgeGradle3Adapter.logger.debug('Using local copy.');
            }
            const stats = await (0, fs_extra_1.lstat)(localPath);
            const mavenComponents = maven_1.MavenUtil.getMavenComponents(lib.name);
            const properId = maven_1.MavenUtil.mavenComponentsToIdentifier(mavenComponents.group, mavenComponents.artifact, mavenComponents.version, mavenComponents.classifier, extension);
            forgeModule.subModules?.push({
                id: properId,
                name: `Minecraft Forge (${mavenComponents?.artifact})`,
                type: helios_distribution_types_1.Type.Library,
                artifact: this.generateArtifact(libBuf, stats, libRepo.getArtifactUrlByComponents(this.baseUrl, mavenComponents.group, mavenComponents.artifact, mavenComponents.version, mavenComponents.classifier, extension))
            });
        }
        return forgeModule;
    }
}
exports.ForgeGradle3Adapter = ForgeGradle3Adapter;
ForgeGradle3Adapter.logger = LoggerUtil_1.LoggerUtil.getLogger('FG3 Adapter');
ForgeGradle3Adapter.WILDCARD_MCP_VERSION = '${mcpVersion}';
//# sourceMappingURL=ForgeGradle3.resolver.js.map