"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ForgeGradle2Adapter = void 0;
const crypto_1 = require("crypto");
const fs_extra_1 = require("fs-extra");
const helios_distribution_types_1 = require("helios-distribution-types");
const path_1 = require("path");
const LibRepo_struct_1 = require("../../../structure/repo/LibRepo.struct");
const maven_1 = require("../../../util/maven");
const PackXZExtractWrapper_1 = require("../../../util/java/PackXZExtractWrapper");
const versionutil_1 = require("../../../util/versionutil");
const forge_resolver_1 = require("../forge.resolver");
const LoggerUtil_1 = require("../../../util/LoggerUtil");
class ForgeGradle2Adapter extends forge_resolver_1.ForgeResolver {
    constructor(absoluteRoot, relativeRoot, baseUrl, minecraftVersion, forgeVersion, discardOutput, invalidateCache) {
        super(absoluteRoot, relativeRoot, baseUrl, minecraftVersion, forgeVersion, discardOutput, invalidateCache);
    }
    static isForVersion(version, libraryVersion) {
        if (version.getMinor() === 12 && !versionutil_1.VersionUtil.isOneDotTwelveFG2(libraryVersion)) {
            return false;
        }
        return versionutil_1.VersionUtil.isVersionAcceptable(version, [7, 8, 9, 10, 11, 12]);
    }
    async getModule() {
        return this.getForgeByVersion();
    }
    isForVersion(version, libraryVersion) {
        return ForgeGradle2Adapter.isForVersion(version, libraryVersion);
    }
    async getForgeByVersion() {
        const libRepo = this.repoStructure.getLibRepoStruct();
        const targetLocalPath = libRepo.getLocalForge(this.artifactVersion, 'universal');
        ForgeGradle2Adapter.logger.debug(`Checking for forge version at ${targetLocalPath}..`);
        if (!await libRepo.artifactExists(targetLocalPath)) {
            ForgeGradle2Adapter.logger.debug('Forge not found locally, initializing download..');
            await libRepo.downloadArtifactByComponents(this.REMOTE_REPOSITORY, LibRepo_struct_1.LibRepoStructure.FORGE_GROUP, LibRepo_struct_1.LibRepoStructure.FORGE_ARTIFACT, this.artifactVersion, 'universal', 'jar');
        }
        else {
            ForgeGradle2Adapter.logger.debug('Using locally discovered forge.');
        }
        ForgeGradle2Adapter.logger.debug(`Beginning processing of Forge v${this.forgeVersion} (Minecraft ${this.minecraftVersion})`);
        let versionManifestBuf;
        try {
            versionManifestBuf = await this.getVersionManifestFromJar(targetLocalPath);
        }
        catch (err) {
            throw new Error('Failed to find version.json in forge universal jar.');
        }
        const versionManifest = JSON.parse(versionManifestBuf.toString());
        const forgeModule = {
            id: maven_1.MavenUtil.mavenComponentsToIdentifier(LibRepo_struct_1.LibRepoStructure.FORGE_GROUP, LibRepo_struct_1.LibRepoStructure.FORGE_ARTIFACT, this.artifactVersion, 'universal'),
            name: 'Minecraft Forge',
            type: helios_distribution_types_1.Type.ForgeHosted,
            artifact: this.generateArtifact(await (0, fs_extra_1.readFile)(targetLocalPath), await (0, fs_extra_1.lstat)(targetLocalPath), libRepo.getArtifactUrlByComponents(this.baseUrl, LibRepo_struct_1.LibRepoStructure.FORGE_GROUP, LibRepo_struct_1.LibRepoStructure.FORGE_ARTIFACT, this.artifactVersion, 'universal')),
            subModules: []
        };
        const postProcessQueue = [];
        for (const lib of versionManifest.libraries) {
            if (lib.name.startsWith('net.minecraftforge:forge:')) {
                // We've already processed forge.
                continue;
            }
            ForgeGradle2Adapter.logger.debug(`Processing ${lib.name}..`);
            const extension = await this.determineExtension(lib, libRepo);
            const localPath = libRepo.getArtifactById(lib.name, extension);
            const postProcess = extension === 'jar.pack.xz';
            let queueDownload = !await libRepo.artifactExists(localPath);
            let libBuf;
            if (!queueDownload) {
                libBuf = await (0, fs_extra_1.readFile)(localPath);
                // VERIFY HASH
                if (!postProcess) { // Checksums for .pack.xz in the version.json are completely useless.
                    if (lib.checksums != null && lib.checksums.length == 1) {
                        const sha1 = (0, crypto_1.createHash)('sha1').update(libBuf).digest('hex');
                        if (sha1 !== lib.checksums[0]) {
                            ForgeGradle2Adapter.logger.debug('Hashes do not match, redownloading..');
                            queueDownload = true;
                        }
                    }
                }
            }
            else {
                ForgeGradle2Adapter.logger.debug('Not found locally, downloading..');
                queueDownload = true;
            }
            if (queueDownload) {
                await libRepo.downloadArtifactById(lib.url || this.MOJANG_REMOTE_REPOSITORY, lib.name, extension);
                libBuf = await (0, fs_extra_1.readFile)(localPath);
            }
            else {
                ForgeGradle2Adapter.logger.debug('Using local copy.');
            }
            const stats = await (0, fs_extra_1.lstat)(localPath);
            const mavenComponents = maven_1.MavenUtil.getMavenComponents(lib.name);
            const properId = maven_1.MavenUtil.mavenComponentsToIdentifier(mavenComponents.group, mavenComponents.artifact, mavenComponents.version, mavenComponents.classifier, extension);
            forgeModule.subModules?.push({
                id: properId,
                name: `Minecraft Forge (${mavenComponents?.artifact})`,
                type: helios_distribution_types_1.Type.Library,
                artifact: this.generateArtifact(libBuf, stats, libRepo.getArtifactUrlByComponents(this.baseUrl, mavenComponents.group, mavenComponents.artifact, mavenComponents.version, mavenComponents.classifier, extension))
            });
            if (postProcess) {
                postProcessQueue.push({
                    id: properId,
                    localPath
                });
            }
        }
        for (const entry of await this.processPackXZFiles(postProcessQueue)) {
            const el = forgeModule.subModules?.find((element) => element.id === entry.id);
            if (el != null) {
                el.artifact.MD5 = entry.MD5;
            }
            else {
                ForgeGradle2Adapter.logger.error(`Error during post processing, could not update ${entry.id}`);
            }
        }
        return forgeModule;
    }
    async determineExtension(lib, libRepo) {
        if (lib.url == null) {
            return 'jar';
        }
        ForgeGradle2Adapter.logger.debug('Determing extension..');
        const possibleExt = [
            'jar.pack.xz',
            'jar'
        ];
        // Check locally.
        for (const ext of possibleExt) {
            const localPath = libRepo.getArtifactById(lib.name, ext);
            const exists = await libRepo.artifactExists(localPath);
            if (exists) {
                return ext;
            }
        }
        // Check remote.
        for (const ext of possibleExt) {
            const exists = await libRepo.headArtifactById(this.REMOTE_REPOSITORY, lib.name, ext);
            if (exists) {
                return ext;
            }
        }
        // Default to jar.
        return 'jar';
    }
    async processPackXZFiles(processingQueue) {
        if (processingQueue.length == 0) {
            return [];
        }
        const accumulator = [];
        const tempDir = this.repoStructure.getTempDirectory();
        if (await (0, fs_extra_1.pathExists)(tempDir)) {
            await (0, fs_extra_1.remove)(tempDir);
        }
        await (0, fs_extra_1.mkdirs)(tempDir);
        const files = [];
        for (const entry of processingQueue) {
            const tmpFile = (0, path_1.join)(tempDir, (0, path_1.basename)(entry.localPath));
            await (0, fs_extra_1.copy)(entry.localPath, tmpFile);
            files.push(tmpFile);
        }
        ForgeGradle2Adapter.logger.debug('Spawning PackXZExtract.');
        const packXZExecutor = new PackXZExtractWrapper_1.PackXZExtractWrapper();
        await packXZExecutor.extractUnpack(files);
        ForgeGradle2Adapter.logger.debug('All files extracted, calculating hashes..');
        for (const entry of processingQueue) {
            const tmpFileName = (0, path_1.basename)(entry.localPath);
            const tmpFile = (0, path_1.join)(tempDir, tmpFileName.substring(0, tmpFileName.indexOf('.pack.xz')));
            const buf = await (0, fs_extra_1.readFile)(tmpFile);
            accumulator.push({
                id: entry.id,
                MD5: (0, crypto_1.createHash)('md5').update(buf).digest('hex')
            });
        }
        ForgeGradle2Adapter.logger.debug('Complete, removing temp directory..');
        await (0, fs_extra_1.remove)(tempDir);
        return accumulator;
    }
}
exports.ForgeGradle2Adapter = ForgeGradle2Adapter;
ForgeGradle2Adapter.logger = LoggerUtil_1.LoggerUtil.getLogger('FG2 Adapter');
//# sourceMappingURL=ForgeGradle2.resolver.js.map