"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ModuleStructure = void 0;
const minimatch_1 = __importDefault(require("minimatch"));
const crypto_1 = require("crypto");
const fs_extra_1 = require("fs-extra");
const helios_distribution_types_1 = require("helios-distribution-types");
const path_1 = require("path");
const BaseModel_struct_1 = require("../BaseModel.struct");
const ClaritasWrapper_1 = require("../../../util/java/ClaritasWrapper");
class ModuleStructure extends BaseModel_struct_1.BaseModelStructure {
    constructor(absoluteRoot, relativeRoot, structRoot, baseUrl, minecraftVersion, type, untrackedFiles, filter) {
        super(absoluteRoot, relativeRoot, structRoot, baseUrl);
        this.minecraftVersion = minecraftVersion;
        this.type = type;
        this.filter = filter;
        this.crudeRegex = /(.+?)-(.+).[jJ][aA][rR]/;
        this.DEFAULT_VERSION = '0.0.0';
        this.FILE_NAME_BLACKLIST = [
            '.gitkeep'
        ];
        this.untrackedFilePatterns = this.determineUntrackedFiles(structRoot, untrackedFiles);
    }
    async getSpecModel() {
        if (this.resolvedModels == null) {
            this.resolvedModels = await this._doModuleRetrieval(await this._doModuleDiscovery(this.containerDirectory));
        }
        return this.resolvedModels;
    }
    getDefaultGroup() {
        return `generated.${this.type.toLowerCase()}`;
    }
    generateMavenIdentifier(group, id, version) {
        return `${group}:${id}:${version}@${helios_distribution_types_1.TypeMetadata[this.type].defaultExtension}`;
    }
    attemptCrudeInference(name) {
        const result = this.crudeRegex.exec(name);
        if (result != null) {
            return {
                name: result[1],
                version: result[2]
            };
        }
        else {
            return {
                name: name.substring(0, name.lastIndexOf('.')),
                version: this.DEFAULT_VERSION
            };
        }
    }
    getClaritasGroup(path) {
        return this.claritasResult[path]?.group || this.getDefaultGroup();
    }
    getClaritasExceptions() {
        return [];
    }
    getClaritasType() {
        return null;
    }
    async parseModule(file, filePath, stats) {
        const artifact = {
            size: stats.size,
            url: await this.getModuleUrl(file, filePath, stats)
        };
        const relativeToContainer = filePath.substr(this.containerDirectory.length + 1);
        const untrackedByPattern = this.isFileUntracked(relativeToContainer);
        if (!untrackedByPattern) {
            const buf = await (0, fs_extra_1.readFile)(filePath);
            artifact.MD5 = (0, crypto_1.createHash)('md5').update(buf).digest('hex');
        }
        else {
            this.logger.debug(`File ${relativeToContainer} is untracked. Matching pattern: ${untrackedByPattern}`);
        }
        const mdl = {
            id: await this.getModuleId(file, filePath),
            name: await this.getModuleName(file, filePath),
            type: this.type,
            artifact
        };
        const pth = await this.getModulePath(file, filePath, stats);
        if (pth) {
            mdl.artifact.path = pth;
        }
        return mdl;
    }
    async _doModuleDiscovery(scanDirectory) {
        const moduleCandidates = [];
        if (await (0, fs_extra_1.pathExists)(scanDirectory)) {
            const files = await (0, fs_extra_1.readdir)(scanDirectory);
            for (const file of files) {
                const filePath = (0, path_1.resolve)(scanDirectory, file);
                const stats = await (0, fs_extra_1.lstat)(filePath);
                if (stats.isFile()) {
                    if (!this.FILE_NAME_BLACKLIST.includes(file)) {
                        if (this.filter == null || this.filter(file, filePath, stats)) {
                            moduleCandidates.push({ file, filePath, stats });
                        }
                    }
                }
            }
        }
        return moduleCandidates;
    }
    async invokeClaritas(moduleCandidates) {
        if (this.getClaritasType() != null) {
            const claritasExecutor = new ClaritasWrapper_1.ClaritasWrapper(this.absoluteRoot);
            let claritasCandidates = moduleCandidates;
            const exceptionCandidates = [];
            for (const exception of this.getClaritasExceptions()) {
                const exceptionCandidate = moduleCandidates.find((value) => value.file.toLowerCase().indexOf(exception.exceptionName) > -1);
                if (exceptionCandidate != null) {
                    exceptionCandidates.push([exceptionCandidate, exception]);
                    claritasCandidates = claritasCandidates.filter((value) => value.file.toLowerCase().indexOf(exception.exceptionName) === -1);
                }
            }
            this.claritasResult = await claritasExecutor.execute(this.getClaritasType(), this.minecraftVersion, claritasCandidates.map(entry => entry.filePath));
            if (this.claritasResult == null) {
                this.logger.error('Failed to process Claritas result!');
            }
            else {
                for (const [candidate, exception] of exceptionCandidates) {
                    this.claritasResult[candidate.filePath] = exception.proxyMetadata;
                }
            }
        }
    }
    async _doModuleRetrieval(moduleCandidates, options) {
        const accumulator = [];
        if (moduleCandidates.length > 0) {
            // Invoke Claritas and attach result to class.
            await this.invokeClaritas(moduleCandidates);
            // Process Modules
            for (const candidate of moduleCandidates) {
                options?.preProcess?.(candidate);
                const mdl = await this.parseModule(candidate.file, candidate.filePath, candidate.stats);
                options?.postProcess?.(mdl);
                accumulator.push(mdl);
            }
        }
        return accumulator;
    }
    determineUntrackedFiles(targetStructRoot, untrackedFileOptions) {
        if (untrackedFileOptions) {
            return untrackedFileOptions
                .filter(x => x.appliesTo.includes(targetStructRoot))
                .reduce((acc, cur) => acc.concat(cur.patterns), []);
        }
        return [];
    }
    // Will return the matching pattern, undefined if no match.
    isFileUntracked(pathRelativeToContainer) {
        return this.untrackedFilePatterns.find(pattern => (0, minimatch_1.default)(pathRelativeToContainer, pattern));
    }
}
exports.ModuleStructure = ModuleStructure;
//# sourceMappingURL=Module.struct.js.map