"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseMavenRepo = void 0;
const got_1 = __importDefault(require("got"));
const fs_extra_1 = require("fs-extra");
const path_1 = require("path");
const url_1 = require("url");
const maven_1 = require("../../util/maven");
const BaseFileStructure_1 = require("../BaseFileStructure");
const LoggerUtil_1 = require("../../util/LoggerUtil");
class BaseMavenRepo extends BaseFileStructure_1.BaseFileStructure {
    constructor(absoluteRoot, relativeRoot, structRoot) {
        super(absoluteRoot, relativeRoot, structRoot);
    }
    getArtifactById(mavenIdentifier, extension) {
        return (0, path_1.resolve)(this.containerDirectory, maven_1.MavenUtil.mavenIdentifierToString(mavenIdentifier, extension));
    }
    getArtifactByComponents(group, artifact, version, classifier, extension = 'jar') {
        return (0, path_1.resolve)(this.containerDirectory, maven_1.MavenUtil.mavenComponentsToString(group, artifact, version, classifier, extension));
    }
    getArtifactUrlByComponents(baseURL, group, artifact, version, classifier, extension = 'jar') {
        return new url_1.URL((0, path_1.join)(this.relativeRoot, maven_1.MavenUtil.mavenComponentsToString(group, artifact, version, classifier, extension)), baseURL).toString();
    }
    async artifactExists(path) {
        return (0, fs_extra_1.pathExists)(path);
    }
    async downloadArtifactById(url, mavenIdentifier, extension) {
        return this.downloadArtifactBase(url, maven_1.MavenUtil.mavenIdentifierToString(mavenIdentifier, extension));
    }
    async downloadArtifactByComponents(url, group, artifact, version, classifier, extension) {
        return this.downloadArtifactBase(url, maven_1.MavenUtil.mavenComponentsToString(group, artifact, version, classifier, extension));
    }
    async downloadArtifactBase(url, relative) {
        const resolvedURL = new url_1.URL(relative, url).toString();
        return this.downloadArtifactDirect(resolvedURL, relative);
    }
    async downloadArtifactDirect(url, path) {
        BaseMavenRepo.logger.debug(`Downloading ${url}..`);
        const request = await got_1.default.stream.get({ url });
        const localPath = (0, path_1.resolve)(this.containerDirectory, path);
        await (0, fs_extra_1.mkdirs)((0, path_1.dirname)(localPath));
        const writer = (0, fs_extra_1.createWriteStream)(localPath);
        request.pipe(writer);
        // tslint:disable-next-line: no-shadowed-variable
        return new Promise((resolve, reject) => {
            writer.on('finish', () => {
                BaseMavenRepo.logger.debug(`Completed download of ${url}.`);
                resolve();
            });
            writer.on('error', reject);
        });
    }
    async headArtifactById(url, mavenIdentifier, extension) {
        return this.headArtifactBase(url, maven_1.MavenUtil.mavenIdentifierToString(mavenIdentifier, extension));
    }
    async headArtifactByComponents(url, group, artifact, version, classifier, extension) {
        return this.headArtifactBase(url, maven_1.MavenUtil.mavenComponentsToString(group, artifact, version, classifier, extension));
    }
    async headArtifactBase(url, relative) {
        const resolvedURL = new url_1.URL(relative, url).toString();
        try {
            const response = await got_1.default.head({
                url: resolvedURL
            });
            return response.statusCode === 200;
        }
        catch (ignored) {
            return false;
        }
    }
}
exports.BaseMavenRepo = BaseMavenRepo;
BaseMavenRepo.logger = LoggerUtil_1.LoggerUtil.getLogger('BaseMavenRepo');
//# sourceMappingURL=BaseMavenRepo.js.map