"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerStructure = void 0;
const fs_extra_1 = require("fs-extra");
const path_1 = require("path");
const url_1 = require("url");
const VersionSegmentedRegistry_1 = require("../../util/VersionSegmentedRegistry");
const servermeta_1 = require("../../model/nebula/servermeta");
const BaseModel_struct_1 = require("./BaseModel.struct");
const File_struct_1 = require("./module/File.struct");
const Library_struct_1 = require("./module/Library.struct");
const MinecraftVersion_1 = require("../../util/MinecraftVersion");
const SchemaUtil_1 = require("../../util/SchemaUtil");
class ServerStructure extends BaseModel_struct_1.BaseModelStructure {
    constructor(absoluteRoot, baseUrl, discardOutput, invalidateCache) {
        super(absoluteRoot, '', 'servers', baseUrl);
        this.discardOutput = discardOutput;
        this.invalidateCache = invalidateCache;
        this.ID_REGEX = /(.+-(.+)$)/;
        this.SERVER_META_FILE = 'servermeta.json';
    }
    getLoggerName() {
        return 'ServerStructure';
    }
    async getSpecModel() {
        if (this.resolvedModels == null) {
            this.resolvedModels = await this._doSeverRetrieval();
        }
        return this.resolvedModels;
    }
    async createServer(id, minecraftVersion, options) {
        const effectiveId = `${id}-${minecraftVersion}`;
        const absoluteServerRoot = (0, path_1.resolve)(this.containerDirectory, effectiveId);
        const relativeServerRoot = (0, path_1.join)(this.relativeRoot, effectiveId);
        if (await (0, fs_extra_1.pathExists)(absoluteServerRoot)) {
            this.logger.error('Server already exists! Aborting.');
            return;
        }
        await (0, fs_extra_1.mkdirs)(absoluteServerRoot);
        const serverMetaOpts = {};
        if (options.forgeVersion != null) {
            const fms = VersionSegmentedRegistry_1.VersionSegmentedRegistry.getForgeModStruct(minecraftVersion, options.forgeVersion, absoluteServerRoot, relativeServerRoot, this.baseUrl, []);
            await fms.init();
            serverMetaOpts.forgeVersion = options.forgeVersion;
        }
        const serverMeta = (0, SchemaUtil_1.addSchemaToObject)((0, servermeta_1.getDefaultServerMeta)(id, minecraftVersion.toString(), serverMetaOpts), SchemaUtil_1.SchemaTypes.ServerMetaSchema, this.absoluteRoot);
        await (0, fs_extra_1.writeFile)((0, path_1.resolve)(absoluteServerRoot, this.SERVER_META_FILE), JSON.stringify(serverMeta, null, 2));
        const libS = new Library_struct_1.LibraryStructure(absoluteServerRoot, relativeServerRoot, this.baseUrl, minecraftVersion, []);
        await libS.init();
        const mfs = new File_struct_1.MiscFileStructure(absoluteServerRoot, relativeServerRoot, this.baseUrl, minecraftVersion, []);
        await mfs.init();
    }
    async _doSeverRetrieval() {
        const accumulator = [];
        const files = await (0, fs_extra_1.readdir)(this.containerDirectory);
        for (const file of files) {
            const absoluteServerRoot = (0, path_1.resolve)(this.containerDirectory, file);
            const relativeServerRoot = (0, path_1.join)(this.relativeRoot, file);
            if ((await (0, fs_extra_1.lstat)(absoluteServerRoot)).isDirectory()) {
                this.logger.info(`Beginning processing of ${file}.`);
                const match = this.ID_REGEX.exec(file);
                if (match == null) {
                    this.logger.warn(`Server directory ${file} does not match the defined standard.`);
                    this.logger.warn('All server ids must end with -<minecraft version> (ex. -1.12.2)');
                    continue;
                }
                let iconUrl = null;
                // Resolve server icon
                const subFiles = await (0, fs_extra_1.readdir)(absoluteServerRoot);
                for (const subFile of subFiles) {
                    const caseInsensitive = subFile.toLowerCase();
                    if (caseInsensitive.endsWith('.jpg') || caseInsensitive.endsWith('.png')) {
                        iconUrl = new url_1.URL((0, path_1.join)(relativeServerRoot, subFile), this.baseUrl).toString();
                    }
                }
                if (!iconUrl) {
                    this.logger.warn(`No icon file found for server ${file}.`);
                }
                // Read server meta
                const serverMeta = JSON.parse(await (0, fs_extra_1.readFile)((0, path_1.resolve)(absoluteServerRoot, this.SERVER_META_FILE), 'utf-8'));
                const minecraftVersion = new MinecraftVersion_1.MinecraftVersion(match[2]);
                const untrackedFiles = serverMeta.untrackedFiles || [];
                const modules = [];
                if (serverMeta.forge) {
                    const forgeResolver = VersionSegmentedRegistry_1.VersionSegmentedRegistry.getForgeResolver(minecraftVersion, serverMeta.forge.version, (0, path_1.dirname)(this.containerDirectory), '', this.baseUrl, this.discardOutput, this.invalidateCache);
                    // Resolve forge
                    const forgeItselfModule = await forgeResolver.getModule();
                    modules.push(forgeItselfModule);
                    const forgeModStruct = VersionSegmentedRegistry_1.VersionSegmentedRegistry.getForgeModStruct(minecraftVersion, serverMeta.forge.version, absoluteServerRoot, relativeServerRoot, this.baseUrl, untrackedFiles);
                    const forgeModModules = await forgeModStruct.getSpecModel();
                    modules.push(...forgeModModules);
                }
                const libraryStruct = new Library_struct_1.LibraryStructure(absoluteServerRoot, relativeServerRoot, this.baseUrl, minecraftVersion, untrackedFiles);
                const libraryModules = await libraryStruct.getSpecModel();
                modules.push(...libraryModules);
                const fileStruct = new File_struct_1.MiscFileStructure(absoluteServerRoot, relativeServerRoot, this.baseUrl, minecraftVersion, untrackedFiles);
                const fileModules = await fileStruct.getSpecModel();
                modules.push(...fileModules);
                accumulator.push({
                    id: match[1],
                    name: serverMeta.meta.name,
                    description: serverMeta.meta.description,
                    icon: iconUrl,
                    version: serverMeta.meta.version,
                    address: serverMeta.meta.address,
                    minecraftVersion: match[2],
                    ...(serverMeta.meta.discord ? { discord: serverMeta.meta.discord } : {}),
                    mainServer: serverMeta.meta.mainServer,
                    autoconnect: serverMeta.meta.autoconnect,
                    modules
                });
            }
            else {
                this.logger.warn(`Path ${file} in server directory is not a directory!`);
            }
        }
        return accumulator;
    }
}
exports.ServerStructure = ServerStructure;
//# sourceMappingURL=Server.struct.js.map