"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ForgeModStructure17 = void 0;
const node_stream_zip_1 = __importDefault(require("node-stream-zip"));
const stringutils_1 = require("../../../../util/stringutils");
const versionutil_1 = require("../../../../util/versionutil");
const ForgeMod_struct_1 = require("../ForgeMod.struct");
const ClaritasResult_1 = require("../../../../model/claritas/ClaritasResult");
class ForgeModStructure17 extends ForgeMod_struct_1.BaseForgeModStructure {
    constructor(absoluteRoot, relativeRoot, baseUrl, minecraftVersion, untrackedFiles) {
        super(absoluteRoot, relativeRoot, baseUrl, minecraftVersion, untrackedFiles);
        this.forgeModMetadata = {};
    }
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    static isForVersion(version, libraryVersion) {
        return versionutil_1.VersionUtil.isVersionAcceptable(version, [7, 8, 9, 10, 11, 12]);
    }
    isForVersion(version, libraryVersion) {
        return ForgeModStructure17.isForVersion(version, libraryVersion);
    }
    getLoggerName() {
        return 'ForgeModStructure (1.7)';
    }
    async getModuleId(name, path) {
        const fmData = await this.getForgeModMetadata(name, path);
        return this.generateMavenIdentifier(this.getClaritasGroup(path), fmData.modid, fmData.version);
    }
    async getModuleName(name, path) {
        return (0, stringutils_1.capitalize)((await this.getForgeModMetadata(name, path)).name);
    }
    getForgeModMetadata(name, path) {
        return new Promise((resolve, reject) => {
            if (!Object.prototype.hasOwnProperty.call(this.forgeModMetadata, name)) {
                const zip = new node_stream_zip_1.default({
                    file: path,
                    storeEntries: true
                });
                zip.on('error', err => reject(err));
                zip.on('ready', () => {
                    try {
                        const res = this.processZip(zip, name, path);
                        zip.close();
                        resolve(res);
                        return;
                    }
                    catch (err) {
                        zip.close();
                        reject(err);
                        return;
                    }
                });
            }
            else {
                resolve(this.forgeModMetadata[name]);
                return;
            }
        });
    }
    isMalformedVersion(version) {
        // Ex. empty, @VERSION@, ${version}
        return version.trim().length === 0 || version.indexOf('@') > -1 || version.indexOf('$') > -1;
    }
    processZip(zip, name, path) {
        // Optifine is a tweak that can be loaded as a forge mod. It does not
        // appear to contain a mcmod.info class. This a special case we will
        // account for.
        if (name.toLowerCase().indexOf('optifine') > -1) {
            // Read zip for changelog.txt
            let changelogBuf;
            try {
                changelogBuf = zip.entryDataSync('changelog.txt');
            }
            catch (err) {
                throw new Error('Failed to read OptiFine changelog.');
            }
            const info = changelogBuf.toString().split('\n')[0].trim();
            const version = info.split(' ')[1];
            this.forgeModMetadata[name] = ({
                modid: 'optifine',
                name: info,
                version,
                mcversion: version.substring(0, version.indexOf('_'))
            });
            return this.forgeModMetadata[name];
        }
        let raw;
        try {
            raw = zip.entryDataSync('mcmod.info');
        }
        catch (err) {
            // ignored
        }
        if (raw) {
            // Assuming the main mod will be the first entry in this file.
            try {
                const resolved = JSON.parse(raw.toString());
                if (Object.prototype.hasOwnProperty.call(resolved, 'modListVersion')) {
                    this.forgeModMetadata[name] = resolved.modList[0];
                }
                else {
                    this.forgeModMetadata[name] = resolved[0];
                }
            }
            catch (err) {
                this.logger.error(`ForgeMod ${name} contains an invalid mcmod.info file.`);
            }
        }
        else {
            this.logger.warn(`ForgeMod ${name} does not contain mcmod.info file.`);
        }
        const cRes = this.claritasResult[path];
        if (cRes == null) {
            this.logger.error(`Claritas failed to yield metadata for ForgeMod ${name}!`);
            this.logger.error('Is this mod malformated or does Claritas need an update?');
        }
        else {
            switch (cRes.modType) {
                case ClaritasResult_1.ForgeModType_1_7.CORE_MOD:
                    this.logger.info(`CORE_MOD Discovered: ForgeMod ${name} has no @Mod annotation. Metadata inference capabilities are limited.`);
                    break;
                case ClaritasResult_1.ForgeModType_1_7.TWEAKER:
                    this.logger.info(`TWEAKER Discovered: ForgeMod ${name} has no @Mod annotation. Metadata inference capabilities may be limited.`);
                    break;
                case ClaritasResult_1.ForgeModType_1_7.UNKNOWN:
                    this.logger.error(`Jar file ${name} is not a ForgeMod. Is it a library?`);
                    break;
            }
        }
        const claritasId = cRes?.id;
        const claritasVersion = cRes?.version;
        const claritasName = cRes?.name;
        // Validate
        const crudeInference = this.attemptCrudeInference(name);
        if (this.forgeModMetadata[name] != null) {
            const x = this.forgeModMetadata[name];
            if (x.modid == null || x.modid === '' || x.modid === this.EXAMPLE_MOD_ID) {
                x.modid = this.discernResult(claritasId, crudeInference.name.toLowerCase());
                x.name = this.discernResult(claritasName, crudeInference.name);
            }
            if (this.forgeModMetadata[name].version != null) {
                const isMalformedVersion = this.isMalformedVersion(this.forgeModMetadata[name].version);
                if (isMalformedVersion) {
                    x.version = this.discernResult(claritasVersion, crudeInference.version);
                }
            }
            else {
                x.version = this.discernResult(claritasVersion, crudeInference.version);
            }
        }
        else {
            this.forgeModMetadata[name] = ({
                modid: this.discernResult(claritasId, crudeInference.name.toLowerCase()),
                name: this.discernResult(claritasName, crudeInference.name),
                version: this.discernResult(claritasVersion, crudeInference.version)
            });
        }
        return this.forgeModMetadata[name];
    }
}
exports.ForgeModStructure17 = ForgeModStructure17;
//# sourceMappingURL=ForgeMod17.struct.js.map