"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MavenUtil = void 0;
const path_1 = require("path");
const url_1 = require("url");
class MavenUtil {
    static mavenComponentsToIdentifier(group, artifact, version, classifier, extension) {
        return `${group}:${artifact}:${version}${classifier != null ? `:${classifier}` : ''}${extension != null ? `@${extension}` : ''}`;
    }
    static isMavenIdentifier(id) {
        return MavenUtil.ID_REGEX.test(id) || MavenUtil.ID_REGEX_WITH_CLASSIFIER.test(id);
    }
    static getMavenComponents(id, extension = 'jar') {
        if (!MavenUtil.isMavenIdentifier(id)) {
            throw new Error('Id is not a maven identifier.');
        }
        let result;
        if (MavenUtil.ID_REGEX_WITH_CLASSIFIER.test(id)) {
            result = MavenUtil.ID_REGEX_WITH_CLASSIFIER.exec(id);
        }
        else {
            result = MavenUtil.ID_REGEX.exec(id);
        }
        if (result != null) {
            return {
                group: result[1],
                artifact: result[2],
                version: result[3],
                classifier: result[4] || undefined,
                extension: result[5] || extension
            };
        }
        throw new Error('Failed to process maven data.');
    }
    static mavenIdentifierToString(id, extension = 'jar') {
        const tmp = MavenUtil.getMavenComponents(id, extension);
        return MavenUtil.mavenComponentsToString(tmp.group, tmp.artifact, tmp.version, tmp.classifier, tmp.extension);
    }
    static mavenComponentsToString(group, artifact, version, classifier, extension = 'jar') {
        return `${group.replace(/\./g, '/')}/${artifact}/${version}/${artifact}-${version}${classifier != null ? `-${classifier}` : ''}.${extension}`;
    }
    static mavenIdentifierToUrl(id, extension = 'jar') {
        return new url_1.URL(MavenUtil.mavenIdentifierToString(id, extension));
    }
    static mavenComponentsToUrl(group, artifact, version, classifier, extension = 'jar') {
        return new url_1.URL(MavenUtil.mavenComponentsToString(group, artifact, version, classifier, extension));
    }
    static mavenIdentifierToPath(id, extension = 'jar') {
        return (0, path_1.normalize)(MavenUtil.mavenIdentifierToString(id, extension));
    }
    static mavenComponentsToPath(group, artifact, version, classifier, extension = 'jar') {
        return (0, path_1.normalize)(MavenUtil.mavenComponentsToString(group, artifact, version, classifier, extension));
    }
}
exports.MavenUtil = MavenUtil;
MavenUtil.ID_REGEX = /(.+):(.+):([^@]+)()(?:@{1}(.+)$)?/;
MavenUtil.ID_REGEX_WITH_CLASSIFIER = /(.+):(.+):(?:([^@]+)(?:-([a-zA-Z]+)))(?:@{1}(.+)$)?/;
//# sourceMappingURL=maven.js.map